############### ###############
## 13 - analysis - estimations
## Project: CBO
## Author: Kamil Kouhen
## Purpose: Creating forest plots and tables with estimations
## Date of creation: 18/01/2022
############### ###############

### Creating uniform path
path <- here("Output/For Publication/Figures/F_Est_")

# 1. A_Breadth_of_engagement --------------- ###### 

  ### List of index components
  toestimate <- c((CBO_groups_final %>% select(contains("bin_")) %>% colnames))
  
  ## a. Estimation with main outcome var first ####

  hyp1_breadth <- rbind(
    lm(A_Breadth_of_engagement ~ treatment + factor(randomization_block), #Region and security situation
       data = CBO_groups_final,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovHC(., 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_Breadth_of_engagement"), #User-written
    
  ## b. Estimations with index components ####
    
    map_dfr(toestimate, function(var){
      set.seed(seed) 
      lm(eval(parse(text = paste0("CBO_groups_final$", var))) ~ treatment + factor(randomization_block), 
         data = CBO_groups_final,
         na.action(na.omit)) %>% 
        coeftest(., 
                 vcov = vcovBS(., 
                               R = 250)) %>% cbo_regression3(outcome.var = var)})
  )
  
  ## c. Creating variable description ####
  
  hyp1_breadth$Description <- as.vector(c("Breadth of CBO engagement in muni. governance", c(" Has met: Muni. Council", paste0("  ''             ",   #Listing small description for variables in forest plot
                                                                                                                               c("Cadre de Concer.",
                                                                                                                                 "Secretary General",
                                                                                                                                 "Muni. Admin", 
                                                                                                                                 "Mayor", 
                                                                                                                                 "Councilor", 
                                                                                                                                 "Prefecture", 
                                                                                                                                 "Health auth.", 
                                                                                                                                 "Education auth.", 
                                                                                                                                 "Water auth.", 
                                                                                                                                 "Village chiefs / CVD"))), 
                                          c(" Has collected: data on muni. services", 
                                            paste0("  ''                     ", "opinions")), 
                                          c(" Has interviewed service providers", 
                                            " Has presented at village meetings", 
                                            " Has collaborated with oth. CBOs or equiv.", 
                                            " Has organized: multi-stakeholder meet.", 
                                            " ''                        meet. with CBO or equiv.",
                                            " Has created Whatsapp group or equiv.",
                                            " Has organized information campaign",
                                            " Has reached out to media outlets")
  ))
  
  
  ## d. Adding False Discovery Rate p-value adjustment for all index components ####
  
  hyp1_breadth$adj.p.value <- stats::p.adjust( #Only doing p-value adjustments for index components
    (hyp1_breadth %>% 
       mutate(p.value = ifelse(outcome == "A_Breadth_of_engagement", NA, p.value)))$p.value, 
    method = "fdr", nrow(hyp1_breadth)) #Creating a vector of adjusted p-values
  
  ## e. Creating two sub-groups to distinguish aggregated variables and its components ####
  
  hyp1_breadth %<>% mutate(sub = ifelse(outcome == "A_Breadth_of_engagement", "Main indicator", "Components"))  
  
  ## f. Calculating control group means ####
  covariates <- c("")
  hyp1_breadth$control.group.mean <- 
    sapply(c("A_Breadth_of_engagement", toestimate), function(var){
      round(eval(parse(text = paste0(
        "mean(
        (CBO_groups_final %>% 
          filter(treatment == 0) %>%
          drop_na(",  var, ",", covariates,"))$", var, "
          )"))), digits = 2)
    })
  
  ## g. Creating forest plots: creating a fake metanalysis type table in order to be able to make forest plot ####
  png(file = paste0(path, 'H1_', 'A_Breadth_of_engagement1.png'), res = 200, width = 12, height = 6.5, units = "in") #Preparing the export of the graph
  
  ### Creating first plot (just with main var)
  meta::metagen( #Creating meta type object
    TE = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$estimate,
    seTE = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$std.error,
    lower = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$conf.low, #Lower bound of CI
    upper = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$conf.high,
    data = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement")),
    method.ci = "t", #t-test for confidence interval
    studlab = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$Description, #Vector created above
    subgroup = (hyp1_breadth %>% filter(outcome == "A_Breadth_of_engagement"))$sub, 
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "Average Treatment Effect"), 
                 xlab = "ATE & \n 95 percent confidence Interval",
                 subgroup = T,
                 subgroup.hetstat = F,
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 text.addline1 = "",
                 addrows.below.overall = 1)
  
  dev.off()
  
  rm(toestimate)
  
  ### Second plot (to be put right after), with components
  png(file = paste0(path, 'H1_', 'A_Breadth_of_engagement2.png'), res = 200, width = 12, height = 6.5, units = "in") #Preparing the export of the graph
  
  meta::metagen( #Creating meta type object
    TE = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$estimate,
    seTE = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$std.error,
    lower = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$conf.low, #Lower bound of CI
    upper = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$conf.high,
    data = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement")),
    method.ci = "t", #t-test for confidence interval
    studlab = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$Description, #Vector created above
    subgroup = (hyp1_breadth %>% filter(outcome != "A_Breadth_of_engagement"))$sub, 
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "Change in Proportion"), 
                 xlab = "Change in proportion & \n 95 percent confidence Interval",
                 subgroup = T,
                 subgroup.hetstat = F,
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 addrows.below.overall = 1)
  
  dev.off()

# 2. A_Intensity_of_engagement --------------- ######
  
  ### List of index components
  toestimate <- c("meet_mayor_num", "meet_councilor_num", "meet_cm", "meet_cdc", "meet_sg_num", "meet_admin_num", "meet_health_total", "meet_education_total", "meet_water_total",  "organize_stakeholder_num", "organize_collab_num", "organize_villagemeeting_num", "pressure_campaign_num")
  
  ## a. Estimation with main outcome var first ####

  hyp1_intensity <- rbind(
    lm(A_Intensity_of_engagement ~ treatment + factor(randomization_block), #Region and security situation
       data = CBO_groups_final,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovHC(., 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_Intensity_of_engagement"), #User-written
    
    ## b. Estimations with index components ####
    
    map_dfr(toestimate, function(var){
      set.seed(seed) 
      #Standardized outcome variable
      lm(scale(eval(parse(text = paste0("CBO_groups_final$", var)))) ~ treatment + factor(randomization_block), 
         data = CBO_groups_final,
         na.action(na.omit)) %>% 
        coeftest(., 
                 vcov = vcovBS(., 
                               R = 250)) %>% cbo_regression3(outcome.var = var)}),
    
    ## c. Number of yearly DM/CBO interactions ####
    
    lm(scale(A_DM_freqinter) ~ treatmentcommune + factor(randomization_block) + #Region and security situation 
         factor(respondent_type), #Controlling for respondent type
       data = DM_finalx2,
       weights = DM_weight,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovCL(., 
                             #Clustering at CBO level
                             cluster = factor(DM_finalx2$appcode), #CBO id 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_DM_freqinter")
  )

  ## d. Need to create vector with small variable description for plot ####
  hyp1_intensity$Description <- as.vector(c("Average: intensity of engagement", c("Nb. of meetings with: elected - Mayor", 
                                                               "''                                    Councilor", 
                                                               "''                                    Muni. Council", 
                                                               "''                                    Cadre de Concert.", 
                                                               "''                          unelected - Secretary General",
                                                               "''                          unelected - Muni. Admin",
                                                               "''                          service providers - health",
                                                               "''                          service providers - education",
                                                               "''                          service providers - water",
                                                               "Nb. of organized multi-stake. meet.",
                                                               "Nb. of CBOs or equiv. collaborated with", 
                                                               "Nb. of villages in which CBO presented", 
                                                               "Nb. of organized information campaign"), 
                         "Nb. of yearly DM/CBO interactions"))

  ## e. Adding False Discovery Rate p-value adjustment for all index components ####
  hyp1_intensity$adj.p.value <- stats::p.adjust( #Only doing p-value adjustments for index components
    (hyp1_intensity %>% 
       mutate(p.value = ifelse(outcome == "A_Intensity_of_engagement" | outcome == "A_DM_freqinter", NA, p.value)))$p.value, 
    method = "fdr", nrow(hyp1_intensity)) #Creating a vector of adjusted p-values

  ## f. Creating three sub-groups to distinguish aggregated variables and its components, and A_DM_freqinter ####
  hyp1_intensity %<>% mutate(sub = ifelse(outcome != "A_DM_freqinter", "According to CBOs", "According to Decision Makers"))

  ## g. Calculating control group means ####
  covariates <- c("")
  hyp1_intensity$control.group.mean <- 
    c(
      round(
        mean((CBO_groups_final %>% 
                filter(treatment == 0) %>%
                drop_na(A_Intensity_of_engagement))$A_Intensity_of_engagement), 
        digits = 2),
      
      sapply(c(toestimate), function(var){
        round(eval(parse(text = paste0(
          "mean(
          (CBO_groups_final %>% 
            drop_na(",  var, ",", covariates,") %>%
            mutate(", var, "= scale(", var, ")) %>%
            filter(treatment == 0))$", var, ")"))), 
          digits = 2)
      }), 
      
      round(
        mean((DM_finalx2 %>% 
                drop_na(A_DM_freqinter, respondent_type) %>%
                mutate(A_DM_freqinter = scale(A_DM_freqinter)) %>%
                filter(treatmentcommune == 0))$A_DM_freqinter), 
        digits = 2)
    )
  
  ## h. Creating forest plot: creating a fake metanalysis type table in order to be able to make forest plot ####

  ###Preparing the export of the graph
  png(file = paste0(path, 'H1_', 'A_Intensity_of_engagement.png'), res = 200, width = 12, height = 6.5, units = "in") 
  
  meta::metagen( #Creating meta type object
    TE = hyp1_intensity$estimate,
    seTE = hyp1_intensity$std.error,
    lower = hyp1_intensity$conf.low, #Lower bound of CI
    upper = hyp1_intensity$conf.high,
    data = hyp1_intensity,
    method.ci = "t", #t-test for confidence interval
    studlab = hyp1_intensity$Description, #Vector created above
    subgroup = hyp1_intensity$sub, 
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "Standardized Average \n Treatment Effect"), 
                 xlab = "Standardized ATE & \n 95 percent confidence interval",
                 subgroup = T,
                 subgroup.hetstat = F,
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 addrows.below.overall = 1)
  
  dev.off()


# 3. A_DM_know_and_inter --------------- ###### 
  
  ### List of respondent type
  toestimate <- c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General")
  
  ## a. Estimation with main outcome var first ####
  
  tab_salience <- rbind(
    lm(A_DM_know_and_inter ~ treatmentcommune + factor(randomization_block) + #Region and security situation
         factor(respondent_type), #Controlling for respondent type
       data = DM_finalx2,
       weights = DM_weight,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovCL(., 
                             #Clustering at CBO level
                             cluster = factor(DM_finalx2$appcode), #CBO id 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_DM_know_and_inter"), 
    
    ## b. Estimations by respondent type  ####
    
    map_dfr(toestimate, function(restype){
      set.seed(seed) 
      #Standardized outcome variable
      lm(A_DM_know_and_inter ~ treatmentcommune + factor(randomization_block), 
         data = DM_finalx2 %>% filter(respondent_type == restype),
         na.action(na.omit)) %>% 
        coeftest(., 
                 vcov = vcovHC(., 
                               type = "HC2")) %>% cbo_regression3(outcome.var = restype)})
  )
  
  ## d. Need to create vector with small variable description for plot ####
  tab_salience$Description <- as.vector(c("All", 
                       "Mayors",
                       "Deputy Mayors",
                       "Opposition Leaders",
                       "Secretary Generals"))

  ## e. Calculating control group means ####
  covariates <- c(", respondent_type")
  tab_salience$control.group.mean <- 
    c(
      round(eval(parse(text = paste0(
        "mean(
        (DM_finalx2 %>% 
          filter(treatmentcommune == 0) %>%
          drop_na(A_DM_know_and_inter", covariates, "))$A_DM_know_and_inter
          )"))), digits = 2),
      sapply(toestimate, function(restype){
        round(eval(parse(text = paste0(
          "mean(
        (DM_finalx2 %>% 
          filter(treatmentcommune == 0) %>%
          drop_na(A_DM_know_and_inter", covariates, ") %>%
          filter(respondent_type =='", restype, "'))$A_DM_know_and_inter
          )"))), digits = 2)
      })
    )
  
  ## f. Number of clusters by group ####
  clustersn <- DM_finalx2 %>% drop_na(A_DM_know_and_inter) %>% distinct(commune) %>% nrow

  tab_salience <- as.data.frame(tab_salience) #Only was direct.textable() works

  direct.textable("
      \\begin{tabularx}{\\textwidth}{l X X X X X}
      \\hline
      & \\multicolumn{5}{c}{\\emph{DV: Awareness of and interaction with CBOs}}\\\\
      \\\\
      & (1) & (2) & (3) & (4) & (5) \\\\
      &  All & Mayors & Deputy Mayors & Opposition Leaders & Secretary General\\\\
      \\hline 
      \\\\
      Control group mean & <<tab_salience[1,'control.group.mean']>> & <<tab_salience[2,'control.group.mean']>> & <<tab_salience[3,'control.group.mean']>> & <<tab_salience[4,'control.group.mean']>> & <<tab_salience[5,'control.group.mean']>> \\\\
      \\\\
      Average Treatment Effect & <<tab_salience[1,'estimate']>> & <<tab_salience[2,'estimate']>> & <<tab_salience[3,'estimate']>> & <<tab_salience[4,'estimate']>> & <<tab_salience[5,'estimate']>> \\\\
      & (<<tab_salience[1,'std.error']>>) & (<<tab_salience[2,'std.error']>>) & (<<tab_salience[3,'std.error']>>) & (<<tab_salience[4,'std.error']>>) & (<<tab_salience[5,'std.error']>>) \\\\
      \\\\
      \\hline
      Observations & <<tab_salience[1,'observations']>> & <<tab_salience[2,'observations']>> & <<tab_salience[3,'observations']>> & <<tab_salience[4,'observations']>> & <<tab_salience[5,'observations']>> \\\\
      Clusters (Municipalities) & <<clustersn>>  & & & & \\\\
      \\hline \\hline
      \\end{tabularx}
      \\footnotesize{Coefficients from OLS regressions with fixed effects for randomization blocks. Standard errors in parentheses, adjusted for clustering by municipality in column (1). $^{*} p<0.1$, $^{**} p<0.05$, $^{***} p<0.01$ (two-sided).}
                  ", 
                  file = here('Output/For Publication/Tables/T_Est_H1_Salience.tex'))
  
  rm(clustersn)

# 5. Hypothesis 1 summary --------------- ###### 

  ## a. Estimations with standardized outcome variables ####
  
  hyp1summary <- bind_rows( 
  
    lm(scale(A_Breadth_of_engagement) ~ treatment + factor(randomization_block), #Region and security situation
       data = CBO_groups_final,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovHC(., 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_Breadth_of_engagement") %>% #User-written
      mutate(Description = "Breadth of CBO engagement in muni. governance"),
    
    lm(A_Intensity_of_engagement ~ treatment + factor(randomization_block), #Region and security situation
       data = CBO_groups_final,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovHC(., 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_Intensity_of_engagement") %>% #User-written
      mutate(Description = "Intensity of CBO engagement in muni. governance"),
    
    lm(scale(A_DM_know_and_inter) ~ treatmentcommune + factor(randomization_block) + #Region and security situation
         factor(respondent_type), #Controlling for respondent type
       data = DM_finalx2,
       weights = DM_weight,
       na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovCL(., 
                             #Clustering at CBO level
                             cluster = factor(DM_finalx2$appcode), #CBO id 
                             type = "HC2")) %>% cbo_regression3(outcome.var = "A_DM_know_and_inter") %>% #User-written
      mutate(Description = "Decision makers' knowledge of and interaction with CBOs")
  )

  ## b. Adding False Discovery Rate p-value adjustment for all main outcome vars ####
  hyp1summary$adj.p.value <- stats::p.adjust(hyp1summary$p.value, 
                                             method = "fdr", nrow(hyp1summary)) #Creating a vector of adjusted p-values
  
  ## c. Calculating control group means ####
  hyp1summary$control.group.mean <- 
    c(
      round(
        mean((CBO_groups_final %>% 
                drop_na(A_Breadth_of_engagement) %>%
                mutate(A_Breadth_of_engagement = scale(A_Breadth_of_engagement)) %>%
                filter(treatment == 0))$A_Breadth_of_engagement), 
        digits = 2), 
      round(
        mean((CBO_groups_final %>% 
                drop_na(A_Intensity_of_engagement) %>%
                mutate(A_Intensity_of_engagement = scale(A_Intensity_of_engagement)) %>%
                filter(treatment == 0))$A_Intensity_of_engagement), 
        digits = 2), 
      round(
        mean((DM_finalx2 %>% 
                drop_na(A_DM_know_and_inter, respondent_type) %>%
                mutate(A_DM_know_and_inter = scale(A_DM_know_and_inter)) %>%
                filter(treatmentcommune == 0))$A_DM_know_and_inter, na.rm = T), 
        digits = 2)
    )

  ### Creating forest plot: creating a fake metanalysis type table in order to be able to make forest plot
  png(file = paste0(path, 'H1_', 'summary.png'), res = 200, width = 12, height = 6.5, units = "in") #Preparing the export of the graph
  
  meta::metagen( #Creating meta type object
    TE = hyp1summary$estimate,
    seTE = hyp1summary$std.error,
    lower = hyp1summary$conf.low, #Lower bound of CI
    upper = hyp1summary$conf.high,
    data = hyp1summary,
    method.ci = "t", #t-test for confidence interval
    studlab = hyp1summary$Description, #Vector created above
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "Average Treatment Effect \n (standardized)"), 
                 xlab = "Standardized ATE & \n 95 percent confidence interval",
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 addrows.below.overall = 1)
  
  dev.off()

# 6. Hypothesis 2 summary --------------- ###### 

  hyp2_sumresults <- bind_rows(
  
  ## a. Estimation: Public goods game - proportion of endowment contributed ####
  # Tobit regression with cluster-robust standard errors 
  # at the CBO level
  AER::tobit(A_game_group_contrib ~ treatment + factor(randomization_block), #Region and security situation
             data = CBO_groups_final_IND, #Individual level dataset derived from CBO level dataset
             weights = IND_weight, # Weights to compensate for th
             # unequal number of interviewee in each CBO
             na.action(na.omit),
             dist = "gaussian", 
             left = 0,
             right = 1) %>%
    coeftest(., 
             vcov = vcovCL(., 
                           #Clustering at CBO level
                           cluster = factor((CBO_groups_final_IND %>% 
                                               drop_na(A_game_group_contrib))$appcode), #CBO id
                           type = "HC0")) %>% cbo_regression3(outcome.var = "A_game_group_contrib") %>%
    mutate(Description = "Public goods game: proportion of endowment contributed") %>%
    mutate(control.group.mean = mean((CBO_groups_final_IND %>%
                                        drop_na(A_game_group_contrib) %>%
                                        filter(treatment == 0))$A_game_group_contrib, na.rm = T)), 
  
  ## b. Estimation: the number of general membership meetings: ####
  lm(meetings_general ~ treatment + factor(randomization_block), #Region and security situation
            data = CBO_groups_final, #CBO level dataset
            na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovHC(., 
                           type = "HC2")) %>% cbo_regression3(outcome.var = "meetings_general") %>% 
    mutate(Description = "General membership meetings held by the CBO in 2020")  %>%
    mutate(control.group.mean = mean((CBO_groups_final %>%
                                        drop_na(meetings_general) %>%
                                        filter(treatment == 0))$meetings_general, na.rm = T)),
  
  ## c. Estimation: the number of executive meetings: ####
  lm(meetings_executive ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_groups_final, #CBO level dataset
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovHC(., 
                           type = "HC2")) %>% cbo_regression3(outcome.var = "meetings_executive") %>%
    mutate(Description = "Executive board meetings held by the CBO in 2020") %>%
    mutate(control.group.mean = mean((CBO_groups_final %>%
                                        drop_na(meetings_executive) %>%
                                        filter(treatment == 0))$meetings_executive, na.rm = T)))
  
  ## d. Estimation: cbo had a budget in 2020 ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp2_sumresults <- bind_rows(hyp2_sumresults,
    
  lm(as.numeric(as.character(budget_any)) ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_groups_final, #CBO level dataset
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovBS(., 
                           R = 250)) %>% cbo_regression3(outcome.var = "budget_any") %>%
    mutate(Description = "CBO had a formal budget in 2020") %>%
    mutate(control.group.mean = mean(as.numeric(as.character((CBO_groups_final %>%
                                        drop_na(budget_any) %>%
                                        filter(treatment == 0))$budget_any)), na.rm = T)),
  
  ## e. Estimation: net membership change ####
  lm(A_log_members_total_after ~ treatment + factor(randomization_block) + #Region and security situation
       A_log_members_total_before, 
     data = CBO_groups_final, #CBO level dataset
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovHC(., 
                           type = "HC2")) %>% cbo_regression3(outcome.var = "A_log_members_total_after") %>%
    mutate(Description = "Net membership change") %>%
    mutate(estimate = exp(estimate) - 1) %>% #De-loging estimates
    mutate(std.error = exp(std.error) - 1) %>% #If need de-logged
    mutate(conf.low = exp(conf.low) - 1) %>% #If need de-logged
    mutate(conf.high = exp(conf.high) - 1) %>% #If need de-logged
    mutate(control.group.mean = exp(mean((CBO_groups_final %>%
                                            drop_na(A_log_members_total_after, A_log_members_total_before) %>%
                                            filter(treatment == 0))$A_log_members_total_after, na.rm = T))))
  
  ## f. Estimation: CBO has adopted COVID-19 policy ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp2_sumresults <- bind_rows(hyp2_sumresults,
    
  lm(as.numeric(as.character(covid_policy)) ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_groups_final, #CBO level dataset
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovBS(., 
                           R = 250)) %>% cbo_regression3(outcome.var = "covid_policy") %>%
    mutate(Description = "CBO has adopted COVID-19 policy") %>%
    mutate(control.group.mean = mean(as.numeric(as.character((CBO_groups_final %>%
                                        drop_na(covid_policy) %>%
                                        filter(treatment == 0))$covid_policy)), na.rm = T)),
    
  ## g. Estimation: CBO has seized COVID-related opportunities ####
  lm(A_covid_adaptation ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_groups_final, #CBO level dataset
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovHC(., 
                           type = "HC2")) %>% cbo_regression3(outcome.var = "A_covid_adaptation") %>% 
    mutate(Description = "CBO has seized COVID-related opportunities") %>%
    mutate(control.group.mean = mean((CBO_groups_final %>%
                                        drop_na(A_covid_adaptation) %>%
                                        filter(treatment == 0))$A_covid_adaptation, na.rm = T)))
  
  
  ## h. Estimation: Perceived internal conflict ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp2_sumresults <- bind_rows(hyp2_sumresults,
  
  lm(A_cbo_conflict ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_individuals_final, #CBO level dataset
     weights = IND_weight,
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovBS(., 
                           cluster = factor(CBO_individuals_final$appcode),
                           R = 250)) %>% cbo_regression3(outcome.var = "A_cbo_conflict") %>%
    mutate(Description = "Perceived internal conflict") %>%
    mutate(control.group.mean = mean((CBO_individuals_final %>%
                                       drop_na(A_cbo_conflict) %>%
                                       filter(treatment == 0))$A_cbo_conflict, na.rm = T)), #Control group mean
  
  ## i. Estimation: Satisfaction with the CBO leadership ####
  lm(as.numeric(as.character(cbo_satisfaction)) ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_individuals_final, #CBO level dataset
     weights = IND_weight,
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovCL(., 
                           cluster = factor(CBO_individuals_final$appcode),
                           type = "HC2")) %>% cbo_regression3(outcome.var = "cbo_satisfaction") %>%
    mutate(Description = "Satisfaction with the CBO leadership") %>%
    mutate(control.group.mean = mean(as.numeric(as.character((CBO_individuals_final %>%
                                        drop_na(cbo_satisfaction) %>%
                                        filter(treatment == 0))$cbo_satisfaction)), na.rm = T)),
  
  ## j. Estimation: Enthusiasm about being a member ####
  lm(as.numeric(as.character(cbo_enthusiasm)) ~ treatment + factor(randomization_block), #Region and security situation
     data = CBO_individuals_final, #CBO level dataset
     weights = IND_weight,
     na.action(na.omit)) %>%
    coeftest(., 
             vcov = vcovCL(., 
                           cluster = factor(CBO_individuals_final$appcode),
                           type = "HC2")) %>% cbo_regression3(outcome.var = "cbo_enthusiasm") %>%
    mutate(Description = "Enthusiasm about being a member") %>%
    mutate(control.group.mean = mean(as.numeric(as.character((CBO_individuals_final %>%
                                        drop_na(cbo_enthusiasm) %>%
                                        filter(treatment == 0))$cbo_enthusiasm)), na.rm = T)))
  
  
  ## k. Estimation: Perceived capture of CBO’s activities ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp2_sumresults <- bind_rows(hyp2_sumresults,
                       
  lm(as.numeric(as.character(cbo_capture)) ~ treatment + factor(randomization_block), #Region and security situation
    data = CBO_individuals_final, #CBO level dataset
    weights = IND_weight,
    na.action(na.omit)) %>%
   coeftest(., 
            vcov = vcovBS(., 
                          cluster = factor(CBO_individuals_final$appcode),
                          R = 250)) %>% cbo_regression3(outcome.var = "cbo_capture") %>%
    mutate(Description = "Perceived capture of CBO’s activities") %>%
    mutate(control.group.mean = mean(as.numeric(as.character((CBO_individuals_final %>%
                                                                drop_na(cbo_capture) %>%
                                                                filter(treatment == 0))$cbo_capture)), na.rm = T)))
  
  ## l. Adding False Discovery Rate p-value adjustment ####
  hyp2_sumresults$adj.p.value <- stats::p.adjust( 
  hyp2_sumresults$p.value, method = "fdr", nrow(hyp2_sumresults)) #Creating a vector of adjusted p-values

  ## m. Creating sub-groups ####
  hyp2_sumresults %<>% mutate(sub = ifelse(row_number() < 2, "Collective Action Capacity", 
                                           ifelse(row_number() >= 2 & row_number() <= 7, "Organizational Capacity", 
                                                  ifelse(row_number() >= 8 & row_number() <= 10, "Internal cohesion", 
                                                         ifelse(row_number() == 11, "Organizational capture", "")))))
  
  ## n. Hyp2 forest plot ####
  png(file = paste0(path, 'H2_', 'summary.png'), res = 200, width = 12, height = 6.5, units = "in") #Preparing the export of the graph

  meta::metagen( #Creating meta type object
    TE = hyp2_sumresults$estimate,
    seTE = hyp2_sumresults$std.error,
    lower = hyp2_sumresults$conf.low, #Lower bound of CI
    upper = hyp2_sumresults$conf.high,
    data = hyp2_sumresults,
    method.ci = "t", #t-test for confidence interval
    studlab = hyp2_sumresults$Description, #Vector created above
    subgroup = hyp2_sumresults$sub, 
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "Average Treatment \n Effect"), 
                 xlab = "ATE & \n 95 percent confidence interval",
                 subgroup = T,
                 subgroup.hetstat = F,
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 addrows.below.overall = 1)
  
  dev.off()

# 7. Hypothesis 3 summary --------------- ###### 

  ## a. Estimation: A_bin_decideurs_important ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp3_sumresults <- 
   lm(A_bin_decideurs_important ~ treatmentcommune + factor(situation) + factor(region) + #Region and security situation
        factor(respondent_type), #Controlling for respondent type
      data = DM_final, #CBO level dataset
      weights = DM_weight,
      na.action(na.omit)) %>%
     coeftest(., 
              vcov = vcovBS(., 
                            cluster = factor(DM_final$commune),
                            R = 250)) %>% cbo_regression3(outcome.var = "A_bin_decideurs_important") %>%
     mutate(Description = "...most important actors to keep happy in order to maintain influence") %>%
     mutate(control.group.mean = mean((DM_final %>%
                                         drop_na(A_bin_decideurs_important) %>%
                                         filter(treatmentcommune == 0))$A_bin_decideurs_important, na.rm = T))
                     
  ## b. Estimation: A_bin_decideurs_justify ####
  set.seed(seed) #Need to set seed for bootstrapping
  
  hyp3_sumresults <- bind_rows(hyp3_sumresults, 
   lm(A_bin_decideurs_justify ~ treatmentcommune + factor(randomization_block) + #Region and security situation
        factor(respondent_type), #Controlling for respondent type
      data = DM_final, #CBO level dataset
      weights = DM_weight,
      na.action(na.omit)) %>%
     coeftest(., 
              vcov = vcovBS(., 
                            cluster = factor(DM_final$commune),
                            R = 250)) %>% cbo_regression3(outcome.var = "A_bin_decideurs_justify") %>%
     mutate(Description = "... actors to whom municipal decision makers spend the most effort justifying their decisions") %>%
     mutate(control.group.mean = mean((DM_final %>%
                                         drop_na(A_bin_decideurs_justify) %>%
                                         filter(treatmentcommune == 0))$A_bin_decideurs_justify, na.rm = T)),                          
   
   ## c. Estimation: pressure_civilsociety ####
   lm(as.numeric(as.character(pressure_civilsociety)) ~ treatmentcommune + factor(randomization_block) + #Region and security situation
        factor(respondent_type), #Controlling for respondent type
      data = DM_final, #CBO level dataset
      weights = DM_weight,
      na.action(na.omit)) %>%
     coeftest(., 
              vcov = vcovHC(., 
                            type = "HC2")) %>% cbo_regression3(outcome.var = "pressure_civilsociety") %>%
     mutate(Description = "Perceived likelihood of civil society pressure if their municipality fails to deliver on its
responsibilities without good justification") %>%
     mutate(control.group.mean = mean(as.numeric(as.character((DM_final %>%
                                         drop_na(pressure_civilsociety) %>%
                                         filter(treatmentcommune == 0))$pressure_civilsociety)), na.rm = T))                             
   )

  #Adding False Discovery Rate p-value adjustment
  hyp3_sumresults$adj.p.value <- stats::p.adjust( 
    hyp3_sumresults$p.value, method = "fdr", nrow(hyp3_sumresults)) #Creating a vector of adjusted p-values
  
  #Creating two sub-groups to distinguish aggregated variables and its components
  hyp3_sumresults %<>% mutate(sub = ifelse(row_number() < 3, "Community groups and civil society organizations are among the three...", 
                                           ifelse(row_number() == 3, "", ""))) 
  
  ##Creating forest plot: creating a fake metanalysis type table in order to be able to make forest plot
  png(file = paste0(path, 'H3_', 'summary.png'), res = 200, width = 17, height = 6.5, units = "in") #Preparing the export of the graph
  
  meta::metagen( #Creating meta type object
    TE = hyp3_sumresults$estimate,
    lower = hyp3_sumresults$conf.low, #Lower bound of CI
    upper = hyp3_sumresults$conf.high,
    data = hyp3_sumresults,
    method.ci = "t", #t-test for confidence interval
    studlab = hyp3_sumresults$Description, #Vector created above
    subgroup = hyp3_sumresults$sub, 
    level = 0.95
  ) %>% 
    meta::forest(overall = F, #Creating plot
                 rightcols = c("p.value", "adj.p.value", "observations"),
                 rightlabs = c("P-value", "Adj. P-value", "Observations"),
                 leftcols = c("studlab", "control.group.mean", "estimate"),
                 leftlabs = c("Variable", "Control Group \n Mean", "ATE \n (change in proportion/probablity)"), 
                 subgroup = T,
                 subgroup.hetstat = F,
                 xlab = "ATE (change in proportion/probablity) & \n 95 percent confidence interval",
                 fixed = F, #Removing indicator for FE
                 random = F, #Removing indicator for RE
                 subgroup.name = NULL,
                 fontfamily = gpar(fontfamily = "serif"),
                 fontsize = 11,
                 print.I2 = F, 
                 print.tau2 = F, 
                 print.pval.Q = F, 
                 addrows.below.overall = 1)
  
  dev.off()

# 8. Hypothesis 4: A_breadth_awareness_sm_indicators --------------- ###### 
  
  ## a. Estimation: number of performance indicators remembered ####
  
  ### List of respondent type
  toestimate <- c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General")
  
  hyp4_indic <- rbind(
    
    AER::tobit(A_breadth_awareness_sm_indicators ~ treatmentcommune + factor(randomization_block) + #Region and security situation
                 factor(respondent_type),
               data = DM_final, #Individual level dataset derived from CBO level dataset
               weights = DM_weight, # Weights to compensate for th
               # unequal number of interviewee in each CBO
               na.action(na.omit),
               dist = "gaussian", 
               left = 0,
               right = 16) %>%
      coeftest(., 
               vcov = vcovCL(., 
                             #Clustering at CBO level
                             cluster = factor(DM_final$commune), #CBO id
                             type = "HC0")) %>% cbo_regression3(outcome.var = "A_breadth_awareness_sm_indicators") %>%
      mutate(Description = "number of performance indicators remembered") %>%
      mutate(control.group.mean = mean((DM_final %>%
                                          drop_na(A_breadth_awareness_sm_indicators, respondent_type) %>%
                                          filter(treatmentcommune == 0))$A_breadth_awareness_sm_indicators, na.rm = T)),
  
    ## b. Estimation: number of performance indicators remembered (for each respondent type)  ####
    
    map_dfr(toestimate, function(restype){
      AER::tobit(A_breadth_awareness_sm_indicators ~ treatmentcommune + factor(randomization_block),
                 data = DM_final %>% filter(respondent_type == restype), #Individual level dataset derived from CBO level dataset
                 weights = DM_weight, # Weights to compensate for th
                 # unequal number of interviewee in each CBO
                 na.action(na.omit),
                 dist = "gaussian", 
                 left = 0,
                 right = 16) %>%
        coeftest(., 
                 vcov = vcovCL(., 
                               #Clustering at CBO level
                               cluster = factor((DM_final %>% filter(respondent_type == restype))$commune), #CBO id
                               type = "HC0")) %>% cbo_regression3(outcome.var = "A_breadth_awareness_sm_indicators") %>%
        mutate(Description = "number of performance indicators remembered") %>%
        mutate(control.group.mean = mean((DM_final %>%
                                            filter(respondent_type == restype) %>%
                                            drop_na(A_breadth_awareness_sm_indicators) %>%
                                            filter(treatmentcommune == 0))$A_breadth_awareness_sm_indicators, na.rm = T))
      })
  )
  
  hyp4_indic$Description<- as.vector(c("All", 
                                       "Mayors",
                                       "Deputy Mayors",
                                       "Opposition Leaders",
                                       "Secretary Generals"))

  ## c. Number of clusters by group  ####
  clustersn <- DM_final %>% drop_na(A_breadth_awareness_sm_indicators) %>% distinct(commune) %>% nrow

  ## d. Number of observations censored at 0  & 16 ####
  ObsAt0 <- c("All" = nrow(DM_final %>% 
                             drop_na(A_breadth_awareness_sm_indicators, respondent_type) %>%
                             filter(A_breadth_awareness_sm_indicators <= 0)), 
              sapply(c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General"), function(res) 
                nrow(DM_final %>% 
                       drop_na(A_breadth_awareness_sm_indicators) %>%
                       filter(respondent_type == res & A_breadth_awareness_sm_indicators <= 0))))


  ObsAt16 <- c("All" = nrow(DM_final %>% 
                              drop_na(A_breadth_awareness_sm_indicators, respondent_type) %>%
                              filter(A_breadth_awareness_sm_indicators <= 16)), 
               sapply(c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General"), function(res) 
                 nrow(DM_final %>% 
                        drop_na(A_breadth_awareness_sm_indicators) %>%
                        filter(respondent_type == res & A_breadth_awareness_sm_indicators <= 16))))

  hyp4_indic <- as.data.frame(hyp4_indic) #Only was direct.textable() works

  ## e. Creating table ## 
  direct.textable("
      \\begin{tabularx}{\\textwidth}{l X X X X X}
      \\hline
      & \\multicolumn{5}{c}{\\emph{DV: Number of performance indicators remembered}}\\\\
      \\\\
      & (1) & (2) & (3) & (4) & (5) \\\\
      &  All & Mayors & Deputy Mayors & Opposition Leaders & Secretary General\\\\
      \\hline 
      \\\\
      \\emph{Control group mean} & <<hyp4_indic[1,'control.group.mean']>> & <<hyp4_indic[2,'control.group.mean']>> & <<hyp4_indic[3,'control.group.mean']>> & <<hyp4_indic[4,'control.group.mean']>> & <<hyp4_indic[5,'control.group.mean']>> \\\\
      \\\\
      Treatment & <<hyp4_indic[1,'estimate']>> & <<hyp4_indic[2,'estimate']>> & <<hyp4_indic[3,'estimate']>> & <<hyp4_indic[4,'estimate']>> & <<hyp4_indic[5,'estimate']>> \\\\
      & (<<hyp4_indic[1,'std.error']>>) & (<<hyp4_indic[2,'std.error']>>) & (<<hyp4_indic[3,'std.error']>>) & (<<hyp4_indic[4,'std.error']>>) & (<<hyp4_indic[5,'std.error']>>) \\\\
      \\\\
      \\hline
      Observations & <<hyp4_indic[1,'observations']>> & <<hyp4_indic[2,'observations']>> & <<hyp4_indic[3,'observations']>> & <<hyp4_indic[4,'observations']>> & <<hyp4_indic[5,'observations']>> \\\\
      Censored at 0 & <<ObsAt0['All']>>  & <<ObsAt0['Mayor']>>  & <<ObsAt0['Deputy Mayor']>>  & <<ObsAt0['Opposition Member']>>  & <<ObsAt0['Secretary General']>>  \\\\
      Censored at 16 & <<ObsAt16['All']>>  & <<ObsAt16['Mayor']>>  & <<ObsAt16['Deputy Mayor']>>  & <<ObsAt16['Opposition Member']>>  & <<ObsAt16['Secretary General']>>  \\\\
      Clusters (Municipalities) & <<clustersn>>  & & & & \\\\
      \\hline \\hline
      \\end{tabularx}
  \\footnotesize{Coefficients from Tobit regressions with fixed effects for randomization blocks. Standard errors in parentheses, adjusted for clustering by municipality in column (1). $^{*} p<0.1$, $^{**} p<0.05$, $^{***} p<0.01$ (two-sided). }", 
                  file = here('Output/For Publication/Tables/T_Est_H4_Awareness.tex'))
  
  rm(clustersn)
  
# 9. Hypothesis 4: A_abs_deviation_relative_perf --------------- ###### 
  
  ## a. Estimation: sum of absolute deviations between decision makers' perception of and actual muni. performance quintile, across all groups of indicators in 2018 ####
  
  ### List of respondent type
  toestimate <- c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General")
  
  hyp4_dev <- rbind(
    
    lm(A_abs_deviation_relative_perf ~ treatmentcommune + factor(randomization_block) + #Region and security situation
                 factor(respondent_type),
               data = DM_final, #Individual level dataset derived from CBO level dataset
               weights = DM_weight, # Weights to compensate for th
               # unequal number of interviewee in each CBO
               na.action(na.omit)) %>%
      coeftest(., 
               vcov = vcovCL(., 
                             #Clustering at CBO level
                             cluster = factor(DM_final$commune), #CBO id
                             type = "HC0")) %>% cbo_regression3(outcome.var = "A_abs_deviation_relative_perf") %>%
      mutate(Description = "Sum of absolute deviations between decision makers' perception of and actual muni. performance quintile") %>%
      mutate(control.group.mean = mean((DM_final %>%
                                          drop_na(A_abs_deviation_relative_perf, respondent_type) %>%
                                          filter(treatmentcommune == 0))$A_abs_deviation_relative_perf, na.rm = T)),
    
    ## b. Estimation: for each respondent type  ####
    
    map_dfr(toestimate, function(restype){
      lm(A_abs_deviation_relative_perf ~ treatmentcommune + factor(randomization_block), #Region and security situation
         data = DM_final %>% filter(respondent_type == restype), #Individual level dataset derived from CBO level dataset
         weights = DM_weight, # Weights to compensate for th
         # unequal number of interviewee in each CBO
         na.action(na.omit)) %>%
        coeftest(., 
                 vcov = vcovCL(., 
                               #Clustering at CBO level
                               cluster = factor((DM_final %>% filter(respondent_type == restype))$commune), #CBO id
                               type = "HC0")) %>% cbo_regression3(outcome.var = "A_abs_deviation_relative_perf") %>%
        mutate(Description = "Sum of absolute deviations between decision makers' perception of and actual muni. performance quintile") %>%
        mutate(control.group.mean = mean(((DM_final %>% filter(respondent_type == restype)) %>%
                                            drop_na(A_abs_deviation_relative_perf) %>%
                                            filter(treatmentcommune == 0))$A_abs_deviation_relative_perf, na.rm = T))
    })
  )
  
  hyp4_dev$Description<- as.vector(c("All", 
                                       "Mayors",
                                       "Deputy Mayors",
                                       "Opposition Leaders",
                                       "Secretary Generals"))
  
  ## c. Number of clusters by group  ####
  clustersn <- DM_final %>% drop_na(A_breadth_awareness_sm_indicators) %>% distinct(commune) %>% nrow
  
  ## d. Number of observations censored at 0  & 16 ####
  ObsAt0 <- c("All" = nrow(DM_final %>% 
                             drop_na(A_breadth_awareness_sm_indicators, respondent_type) %>%
                             filter(A_breadth_awareness_sm_indicators <= 0)), 
              sapply(c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General"), function(res) 
                nrow(DM_final %>% 
                       drop_na(A_breadth_awareness_sm_indicators) %>%
                       filter(respondent_type == res & A_breadth_awareness_sm_indicators <= 0))))
  
  
  ObsAt16 <- c("All" = nrow(DM_final %>% 
                              drop_na(A_breadth_awareness_sm_indicators, respondent_type) %>%
                              filter(A_breadth_awareness_sm_indicators <= 16)), 
               sapply(c("Mayor", "Deputy Mayor", "Opposition Member", "Secretary General"), function(res) 
                 nrow(DM_final %>% 
                        drop_na(A_breadth_awareness_sm_indicators) %>%
                        filter(respondent_type == res & A_breadth_awareness_sm_indicators <= 16))))
  
  hyp4_dev <- as.data.frame(hyp4_dev) #Only was direct.textable() works 

  
  ## e. Creating table ## 
  
  direct.textable("
      \\begin{tabularx}{\\textwidth}{l X X X X X}
      \\hline
      & \\multicolumn{5}{c}{\\emph{DV: Sum of absolute deviations between decision makers' perception of and actual muni. performance quintile}} \\\\
      \\\\
      & (1) & (2) & (3) & (4) & (5) \\\\
      &  All & Mayors & Deputy Mayors & Opposition Leaders & Secretary General \\\\
      \\hline 
      \\\\
      Control group mean & <<hyp4_dev[1,'control.group.mean']>> & <<hyp4_dev[2,'control.group.mean']>> & <<hyp4_dev[3,'control.group.mean']>> & <<hyp4_dev[4,'control.group.mean']>> & <<hyp4_dev[5,'control.group.mean']>> \\\\
      \\\\
      Treatment & <<hyp4_dev[1,'estimate']>> & <<hyp4_dev[2,'estimate']>> & <<hyp4_dev[3,'estimate']>> & <<hyp4_dev[4,'estimate']>> & <<hyp4_dev[5,'estimate']>> \\\\
      & (<<hyp4_dev[1,'std.error']>>) & (<<hyp4_dev[2,'std.error']>>) & (<<hyp4_dev[3,'std.error']>>) & (<<hyp4_dev[4,'std.error']>>) & (<<hyp4_dev[5,'std.error']>>) \\\\
      \\\\
      \\hline
      Observations & <<hyp4_dev[1,'observations']>> & <<hyp4_dev[2,'observations']>> & <<hyp4_dev[3,'observations']>> & <<hyp4_dev[4,'observations']>> & <<hyp4_dev[5,'observations']>> \\\\
      Clusters (Municipalities) & <<clustersn>>  & & & & \\\\
      \\hline \\hline
      \\end{tabularx}
  \\footnotesize{Average treatment effects estimated via OLS regression with fixed effects for randomization blocks. Standard errors in parentheses, adjusted for clustering by municipality in column (1). $^{*} p<0.1$, $^{**} p<0.05$, $^{***} p<0.01$ (two-sided). }
  ", 
                  file = here('Output/For Publication/Tables/T_Est_H4_Accuracy.tex'))
  
  rm(clustersn)
